<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Apishowroom; // Ensure this is the correct model for API-enabled showrooms
use Illuminate\Support\Facades\Log; // For logging errors or debug information
use App\Models\ProductPrice;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    public function index(Request $request)
    {
        $query = Product::query();

        if ($request->search) {
            $query->where('productname', 'LIKE', '%' . $request->search . '%');
        }

        $products = $query->orderBy('id', 'desc')->paginate(40);
        // Fetch all available API-enabled showrooms
        $apishowrooms = Apishowroom::orderBy('name')->get();

        return view('fac.products.index', compact('products', 'apishowrooms'));
    }

    public function create()
    {
        return view('fac.products.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'dateofentry' => 'required|date',
            'productname' => 'required|string|max:255',
            'productiondepartment' => 'required|string|max:100',
            'productcategory' => 'required|string|max:100',
            'price' => 'required|numeric|min:0'
        ]);

        Product::create($validated);
        return redirect()->route('products.index')->with('success', 'Product created!');
    }

    public function edit(Product $product)
    {
        return view('fac.products.edit', compact('product'));
    }

    public function update(Request $request, Product $product)
    {
        $validated = $request->validate([
            'dateofentry' => 'required|date',
            'productname' => 'required|string|max:255',
            'productiondepartment' => 'required|string|max:100',
            'productcategory' => 'required|string|max:100',
            'price' => 'required|numeric|min:0'
        ]);

        $product->update($validated);
        return redirect()->route('products.index')->with('success', 'Product updated!');
    }

    public function destroy(Product $product)
    {
        $product->delete();
        return redirect()->route('products.index')->with('success', 'Product deleted!');
    }

        public function syncInternalProductToProductPrices(Request $request)
    {
        // Ensure only authenticated users can perform this action
        if (!Auth::check()) {
            return response()->json(['message' => 'Unauthorized action.'], 403);
        }

        $syncDate = Carbon::createFromFormat('Y-m-d', '2025-06-01')->startOfDay(); // Define the sync date

        // Fetch all products
        $products = Product::all();

        $syncedCount = 0;
        $errors = [];

        foreach ($products as $product) {
            try {
                // Find or create a ProductPrice entry for this product and effective_start_date
                // We're looking for a general price, not a showroom-specific override,
                // so we include is_showroom_override = false and showroom_id = NULL in the search criteria.
                $productPrice = ProductPrice::updateOrCreate(
                    [
                        'product_id' => $product->id,
                        'effective_start_date' => $syncDate,
                        'is_showroom_override' => false, // Important: Assume general price sync
                        'showroom_id' => null,            // Important: Assume general price sync
                    ],
                    [
                        'price' => $product->price,
                        'changed_by_user_id' => Auth::id(), // ID of the logged-in user
                        'reason' => 'Initial sync from Products table or price update',
                        'effective_end_date' => null, // Or set to a far future date if you prefer
                    ]
                );
                $syncedCount++;
            } catch (\Exception $e) {
                // Log or collect errors for products that failed to sync
                $errors[] = "Failed to sync product ID {$product->id} ({$product->productname}): " . $e->getMessage();
                \Log::error("ProductPrice sync error: " . $e->getMessage(), ['product_id' => $product->id]);
            }
        }

        if (!empty($errors)) {
            return response()->json([
                'message' => "Synchronization completed with {$syncedCount} successes and " . count($errors) . " errors.",
                'errors' => $errors
            ], 200); // Or 500 if you consider any error a failure
        }

        return response()->json(['message' => "Successfully synced {$syncedCount} product prices."]);
    }
}
